<?php

namespace App\Http\Controllers;

use App\Enums\ActiveEnum;
use App\Http\Controllers\User\UserController;
use App\Http\Resources\UserResource;
use App\Models\Product;
use App\Models\ProductUser;
use App\Models\User;
use http\Env\Response;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;

class ProductUsersController extends Controller
{

    public function index(Request $request, Product $product)
    {
        try {
            $productId = $product->product_id;

            $usersCount = \DB::table('uproducts')
                ->where('product_id', $productId)
                ->where('active', 1)
                ->count();

            if ($usersCount === 0) {
                return response()->json([
                    'success' => true,
                    'users' => [],
                    'message' => 'No users found for this product',
                    'product_id' => $productId
                ]);
            }

            $limit = $request->input('limit', 50);
            $limit = min(max((int) $limit, 1), 1000);

            $offset = $request->input('offset', 0);
            $offset = max((int) $offset, 0);

            $query = DB::table('users')
                ->join('uproducts', 'users.user_id', '=', 'uproducts.user_id')
                ->leftJoin('cities', 'users.city', '=', 'cities.ci_id')
                ->leftJoin('countries', 'cities.co_id', '=', 'countries.co_id')
                ->where('uproducts.product_id', $product->product_id)
                ->where('uproducts.active', 1);

            if ($request->has('country') && $request->input('country')) {
                $query->where('countries.co_id', $request->input('country'));
            }
            if ($request->has('city') && $request->input('city')) {
                $query->where('cities.ci_id', $request->input('city'));
            }
            if ($request->has('gender') && $request->input('gender') !== '') {
                $query->where('users.gender', $request->input('gender'));
            }

            $users = $query->select(
                'users.*',
                'uproducts.active as pivot_active',
                'uproducts.date as pivot_date',
                'uproducts.ldate as pivot_ldate',
                'uproducts.language as pivot_language',
                'cities.citye as city_name_en',
                'cities.city as city_name_ar',
                'countries.country_e as country_name_en',
                'countries.country as country_name_ar',
                'countries.co_id as country_id'
            )
                ->limit((int)$limit)
                ->offset((int)$offset)
                ->orderBy('users.user_id', 'desc')
                ->get();


            $formattedUsers = $users->map(function ($user) {
                return [
                    'user_id' => $user->user_id,
                    'id' => $user->user_id,
                    'username' => $user->username,
                    'fname_ar' => $user->fname_ar,
                    'fname_en' => $user->fname_en,
                    'lname_ar' => $user->lname_ar,
                    'lname_en' => $user->lname_en,
                    'fname' => app()->getLocale() == 'ar' ? $user->fname_ar : $user->fname_en,
                    'lname' => app()->getLocale() == 'ar' ? $user->lname_ar : $user->lname_en,
                    'email' => $user->email,
                    'verified' => $user->verified,
                    'close' => $user->close,
                    'active' => $user->pivot_active,
                    'joined_date' => $user->pivot_date,
                    'last_update' => $user->pivot_ldate,
                    'language' => $user->pivot_language,

                    //
                    'mobile' => $user->mobile,
                    'gender' => $user->gender,
                    'bdate' => $user->bdate,
                    'company' => app()->getLocale() == 'ar' ? $user->company : $user->en_company,
                    'address' => app()->getLocale() == 'ar' ? $user->ar_address : $user->en_address,
                    'image_url' => $user->image_url ?? 'https://assets.bucketlistly.blog/sites/5adf778b6eabcc00190b75b1/content_entry5adf77af6eabcc00190b75b6/6075185986d092000b192d0a/files/best-free-travel-images-main-image-hd-op.jpg',
                    'city_name' => app()->getLocale() == 'ar' ? $user->city_name_ar : $user->city_name_en,
                    'city' => $user->city,
                    'country' => $user->country_id,
                    'city_name' => app()->getLocale() == 'ar' ? $user->city_name_ar : $user->city_name_en,
                    'country_name' => app()->getLocale() == 'ar' ? $user->country_name_ar : $user->country_name_en,
                ];
            });

            return response()->json([
                'success' => true,
                'users' => $formattedUsers,
                'total' => $usersCount,
                'limit' => $limit,
                'offset' => $offset,
                'product_id' => $productId
            ]);
        } catch (\Illuminate\Database\QueryException $e) {
            \Log::error('Database Query Error: ' . $e->getMessage());
            \Log::error('SQL: ' . ($e->getSql() ?? 'N/A'));

            return response()->json([
                'success' => false,
                'message' => 'Database query error',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error',
                'product_id' => $product->product_id ?? null
            ], 500);
        } catch (\Exception $e) {
            \Log::error('ProductUsersController Error: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());

            return response()->json([
                'success' => false,
                'message' => 'Error retrieving users',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    public function debug(Request $request, Product $product)
    {
        try {
            $productId = $product->product_id;

            $info = [
                'product_id' => $productId,
                'request_params' => $request->all(),
                'product_model' => [
                    'id' => $product->product_id,
                    'name' => $product->name ?? 'N/A',
                    'active' => $product->active ?? 'N/A'
                ]
            ];

            $allRelationships = \DB::table('uproducts')
                ->where('product_id', $productId)
                ->count();
            $info['total_relationships'] = $allRelationships;

            $activeRelationships = \DB::table('uproducts')
                ->where('product_id', $productId)
                ->where('active', 1)
                ->count();
            $info['active_relationships'] = $activeRelationships;

            $sampleUsers = \DB::table('users')
                ->join('uproducts', 'users.id', '=', 'uproducts.user_id')
                ->where('uproducts.product_id', $productId)
                ->where('uproducts.active', 1)
                ->select('users.id', 'users.username', 'users.email', 'uproducts.active')
                ->limit(3)
                ->get();
            $info['sample_users'] = $sampleUsers;

            $dbCheck = \DB::select('SELECT VERSION() as version');
            $info['database_version'] = $dbCheck[0]->version ?? 'Unknown';

            return response()->json([
                'success' => true,
                'debug_info' => $info
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'trace' => config('app.debug') ? $e->getTraceAsString() : 'Debug trace hidden'
            ]);
        }
    }

    public function removeUser(Request $request, User $user)
    {
        // && count($user->products) == 1
        if ($request->header('product') == env('PRODUCT_ID')) {
            $user->photos()->delete();
            $user->covers()->delete();
            $user->delete();
        }

        ProductUser::where('user_id', $user->user_id)
            ->where('product_id', $request->header('product'))
            ->delete();
        return response()->json([
            'success' => true,
            'message' => __('User deleted successfully'),
        ]);
    }

    public function insertUsers(Request $request)
    {
        // get user from request
        $users = $request->get('users');
        $product_id = $request->header('product');
        $emails = array_keys($users);

        $newUsers = [];

        $oldUsers = User::whereIn('email', $emails)->get();

        foreach ($oldUsers as $oldUser) {
            $oldUser->products()->attach($product_id);
            unset($users[$oldUser->email]);
        }

        $newUsers = array_values($users);

        $importedUsers = [];
        $createdUsers = [];

        foreach ($newUsers as $newUser) {
            $plaintextPassword = $newUser['password'];
            $newUser['verified'] = 2;
            $importedUsers = User::create($newUser);
            $importedUsers->products()->attach($product_id);

            $importedUsers['newpass'] = $plaintextPassword;

            $createdUsers[] = $importedUsers;
        }

        // success and fail
        return response()->json([
            'status' => 200,
            'success' => true,
            'oldUsers' => $oldUsers,
            'newUsers' => $createdUsers,
        ]);
    }

    public function checkUser(Request $request)
    {
        $users = $request->get('user');
        $email = array_keys($users);

        $user = User::where('email', $email)->first();

        $product_id = $request->header('product');
        $createdUsers = [];

        if ($user) {
            $user->products()->attach($product_id);
        } else {
            $users['password'] = Str::random();
            $users['username'] = $this->generateName($users['email']);

            $plaintextPassword = $users['password'];
            $users['verified'] = 2;
            $importedUser = User::create($users);
            $importedUser->products()->attach($product_id);

            $importedUser['newPassWord'] = $plaintextPassword;
            $createdUsers[] = $importedUser;
        }
        //        return response()->json($createdUsers);

        return response()->json([
            'status' => 200,
            'success' => true,
            'oldUsers' => $user,
            'newUsers' => $createdUsers,
        ]);
    }

    public function generateName($email)
    {
        $username = explode('@', $email)[0];
        $username = preg_replace('/[^A-Za-z0-9]/', '_', $username);
        $username = $username . random_int(100, 999);

        return $username;
    }
}
