<?php

namespace App\Http\Controllers\User;

use App\Constants\UserConstants;
use App\Http\Controllers\Controller;
use App\Http\Requests\CreateUserRequest;
use App\Http\Requests\SetCoverRequest;
use App\Http\Requests\SetMainPhotoRequest;
use App\Http\Requests\UpdateUserRequest;
use App\Http\Resources\ImageResource;
use App\Http\Resources\ProductResource;
use App\Http\Resources\UserPreferenceResource;
use App\Http\Resources\UserResource;
use App\Models\Image;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class UserController extends Controller
{
    public function createUser(CreateUserRequest $request)
    {
        $requestData = $request->validated();
        $requestData['password'] = 'password';
        $requestData['password_confirmation'] = 'password';

        $user = User::create($requestData);
        // edit to get pass
        $user['newpass'] = $requestData['password'];

        if ($user) {
            $user->products()->attach($request->header('product'));
        }
        return response()->json(['user' => $user]);
    }

    public function toggleActivation(User $user)
    {
        if ($user->active == 1) {
            $user->active = 0;
        } else {
            $user->active = 1;
        }

        $user->save();

        return response()->json(['message' => __('User activation active toggled successfully')]);
    }

    public function getUser(Request $request, User $user)
    {
        if (!$user) {
            return response()->json(['error' => 'User not found'], 404);
        }

        $fields = $request->query('fields');
        $fieldsArray = $fields ? explode(',', $fields) : ['*'];
        if (in_array('all', $fieldsArray)) {
            $user->country = $user->cityy()->first()->country->co_id;
            $user->city = $user->city;
            return response()->json(['user' => $user]);
        }
        $fieldsArray = array_map(function ($field) {
            return match ($field) {
                'fname' => app()->getLocale() == 'ar' ? 'fname_ar' : 'fname_en',
                'lname' => app()->getLocale() == 'ar' ? 'lname_ar' : 'lname_en',
                'fa_name' => app()->getLocale() == 'ar' ? 'fa_name_ar' : 'fa_name_en',
                'job' => app()->getLocale() == 'ar' ? 'job' : 'en_job',
                'company' => app()->getLocale() == 'ar' ? 'company' : 'en_company',
                'address' => app()->getLocale() == 'ar' ? 'ar_address' : 'en_address',
                'caddress' => app()->getLocale() == 'ar' ? 'ar_caddress' : 'en_caddress',
                    // 'aboutme' => app()->getLocale() == 'ar' ? 'ar_aboutme' : 'en_aboutme',
                default => $field,
            };
        }, $fieldsArray);

        $fieldsArray[] = 'verified';
        $fieldsArray[] = 'remember_token';

        $user = User::where('email', $user->email)->select($fieldsArray)->first();

        return response()->json(['user' => new UserResource($user)]);
    }

    public function getUserByEmail(Request $request, $email)
    {
        $user = User::where('email', $email)->first();
        if (!$user) {
            return response()->json(['error' => 'User not found'], 404);
        }

        $fields = $request->query('fields');
        $fieldsArray = $fields ? explode(',', $fields) : ['*'];
        $fieldsArray = array_map(function ($field) {
            return match ($field) {
                'fname' => app()->getLocale() == 'ar' ? 'fname_ar' : 'fname_en',
                'lname' => app()->getLocale() == 'ar' ? 'lname_ar' : 'lname_en',
                'fa_name' => app()->getLocale() == 'ar' ? 'fa_name_ar' : 'fa_name_en',
                'job' => app()->getLocale() == 'ar' ? 'job' : 'en_job',
                'company' => app()->getLocale() == 'ar' ? 'company' : 'en_company',
                'address' => app()->getLocale() == 'ar' ? 'ar_address' : 'en_address',
                'caddress' => app()->getLocale() == 'ar' ? 'ar_caddress' : 'en_caddress',
                    // 'aboutme' => app()->getLocale() == 'ar' ? 'ar_aboutme' : 'en_aboutme',
                default => $field,
            };
        }, $fieldsArray);

        $fieldsArray[] = 'verified';

        $user = User::where('email', $user->email)->select($fieldsArray)->first();

        return response()->json(['user' => new UserResource($user)]);
    }

    public function update(UpdateUserRequest $request, User $user)
    {
        // Session::put('update_user', $user);
        Log::info('Hi');
        $requestData = $request->validated();
        if ($request->hasFile('image')) {
            $avatarPath = $request->file('image')->store('covers', 'public');
            $image = Image::create([
                'image' => asset(Storage::url($avatarPath)),
                'user_id' => $user->user_id,
                'add_date' => \now(),
                'type' => UserConstants::IMAGE_TYPE_PROFILE,
            ]);
            $requestData['image'] = $image->img_id;
        }

        if ($request->hasFile('cover')) {
            $avatarPath = $request->file('cover')->store('covers', 'public');
            $image = Image::create([
                'image' => asset(Storage::url($avatarPath)),
                'user_id' => $user->user_id,
                'add_date' => \now(),
                'type' => UserConstants::IMAGE_TYPE_COVER,
            ]);
            $requestData['cover'] = $image->img_id;
        }

        $user->update($requestData);

        return response()->json([
            'success' => true,
            'message' => __('User details updated successfully'),
            'user' => $user
        ]);
    }

    public function search(UpdateUserRequest $request)
    {
        $filters = $request->query();

        $query = User::filter($filters);

        $fields = $request->query('fields');
        $fieldsArray = $fields ? explode(',', $fields) : ['*'];
        $fieldsArray = array_map(function ($field) {
            return match ($field) {
                'fname' => app()->getLocale() == 'ar' ? 'fname_ar' : 'fname_en',
                'lname' => app()->getLocale() == 'ar' ? 'lname_ar' : 'lname_en',
                'fa_name' => app()->getLocale() == 'ar' ? 'fa_name_ar' : 'fa_name_en',
                'job' => app()->getLocale() == 'ar' ? 'job' : 'en_job',
                'company' => app()->getLocale() == 'ar' ? 'company' : 'en_company',
                'address' => app()->getLocale() == 'ar' ? 'ar_address' : 'en_address',
                'caddress' => app()->getLocale() == 'ar' ? 'ar_caddress' : 'en_caddress',
                    // 'aboutme' => app()->getLocale() == 'ar' ? 'ar_aboutme' : 'en_aboutme',
                default => $field,
            };
        }, $fieldsArray);

        $users = $query->select($fieldsArray)->get();

        return response()->json([
            'success' => true,
            'data' => UserResource::collection($users),
        ]);
    }

    public function getPreferences(User $user)
    {
        return response()->json([
            'success' => true,
            'data' => new UserPreferenceResource($user),
        ]);
    }

    public function getUsers(Request $request)
    {
        $userIds = $request->input('ids');
        $users = User::whereIn('user_id', $userIds)->get();

        return response()->json([
            'success' => true,
            'data' => UserResource::collection($users),
        ]);
    }

    public function setCover(SetCoverRequest $request)
    {
        $user = auth()->user();

        if ($request->has('cover_id')) {
            $user->cover = $request->cover_id;
        } else {
            $coverPath = $request->file('cover')->store('covers', 'public');
            $image = Image::create([
                'image' => asset(Storage::url($coverPath)),
                'user_id' => $user->user_id,
                'add_date' => now(),
                'type' => UserConstants::IMAGE_TYPE_COVER,
            ]);

            $user->cover = $image->img_id;
        }

        $user->save();

        return response()->json([
            'success' => true,
            'message' => __('User cover updated successfully'),
        ]);
    }

    public function getCovers()
    {
        return response()->json([
            'success' => true,
            'covers' => ImageResource::collection(Auth::user()->covers),
        ]);
    }

    public function setMainPhoto(SetMainPhotoRequest $request)
    {
        $user = auth()->user();

        if ($request->has('photo_id')) {
            $user->image = $request->photo_id;
        } else {
            $photoPath = $request->file('photo')->store('photos', 'public');
            $image = Image::create([
                'image' => asset(Storage::url($photoPath)),
                'user_id' => $user->user_id,
                'add_date' => now(),
                'type' => UserConstants::IMAGE_TYPE_PROFILE,
            ]);

            $user->image = $image->img_id;
        }

        $user->save();

        return response()->json([
            'success' => true,
            'message' => __('User photo updated successfully'),
        ]);
    }

    public function getPhotos()
    {
        return response()->json([
            'success' => true,
            'photos' => ImageResource::collection(Auth::user()->photos),
        ]);
    }

    public function getUserImages(User $user)
    {
        $images = Image::where('user_id', $user->user_id)->get();
        return response()->json(['data' => ImageResource::collection($images)]);
    }

    // public function setUserImages(Request $request, User $user)
    // {
    //     $data = $request->all();
    //     foreach ($data['images'] as $img) {
    //         $imageContents = file_get_contents($img);
    //         $filename = uniqid() . '.' . pathinfo($img, PATHINFO_EXTENSION);
    //         Storage::disk('public')->put($filename, $imageContents);
    //         $imageUrl = '/' . $filename;

    //         Image::create([
    //             'image' => $imageUrl,
    //             'user_id' => $user->user_id,
    //             'add_date' => now(),
    //             'type' => $data['type'],
    //         ]);
    //     }
    //     return response()->json(['success' => __('Saved successfully')]);
    // }



    public function setUserImages(Request $request, User $user)
    {
        $data = $request->all();
        $imagesAdded = 0;

        if (!isset($data['images']) || !is_array($data['images'])) {
            return response()->json(['error' => 'Invalid or missing images data.'], 400);
        }

        DB::beginTransaction();
        try {
            foreach ($data['images'] as $imageData) {
                if (!isset($imageData['content']) || !isset($imageData['filename'])) {
                    continue;
                }

                $imageContents = base64_decode($imageData['content']);
                if ($imageContents === false) {
                    continue;
                }

                $directory = ($data['type'] == \App\Constants\UserConstants::IMAGE_TYPE_COVER) ? 'covers' : 'profiles';
                $newFilename = $directory . '/' . $imageData['filename'];

                Storage::disk('public')->put($newFilename, $imageContents);

                $newImage = Image::create([
                    'image' => $newFilename,
                    'user_id' => $user->user_id,
                    'add_date' => now(),
                    'type' => $data['type'],
                ]);


                if ($data['type'] == \App\Constants\UserConstants::IMAGE_TYPE_PROFILE) {

                    $user->image = $newImage->img_id;
                    $user->save();



                }

                $imagesAdded++;
            }

            DB::commit();

            if ($imagesAdded > 0) {
                return response()->json(['success' => __(':count image(s) saved successfully', ['count' => $imagesAdded])]);
            } else {
                return response()->json(['error' => __('Failed to save any images.')], 400);
            }
        } catch (\Exception $e) {
            DB::rollBack(); //
            Log::error('Exception in setUserImages for user: ' . $user->user_id, [
                'error' => $e->getMessage()
            ]);
            return response()->json(['error' => 'An internal error occurred.'], 500);
        }
    }

    public function deleteImage(Image $image)
    {
        $user = User::where('user_id', $image->user_id)->first();

        if ($user->image == $image->img_id) {
            $user->image = null;
            $user->save();
        } else if ($user->cover == $image->img_id) {
            $user->cover = null;
            $user->save();
        }

        $image->delete();

        return response()->json([
            'message' => __('Deleted successfully'),
            'status' => 200,
            'success' => true
        ]);
    }

    public function SetImage(Image $image)
    {
        $user = User::where('user_id', $image->user_id)->first();
        if ($image->type == UserConstants::IMAGE_TYPE_PROFILE) {
            $user->image = $image->img_id;
        } else if ($image->type == UserConstants::IMAGE_TYPE_COVER) {
            $user->cover = $image->img_id;
        }
        $user->save();

        return response()->json(['message' => __('Saved successfully')]);
    }

    public function generateUsername(Request $request)
    {
        $request->validate([
            'email' => 'required|email'
        ]);
        $email = $request->get('email');
        $username = explode('@', $email)[0];
        $username = preg_replace('/[^A-Za-z0-9]/', '_', $username);
        $isValid = false;
        while (!$isValid) {
            $isExist = User::where('username', $username)->exists();
            if (!$isExist) {
                break;
            }
            $username = $username . random_int(100, 999);
        }

        return response()->json([
            'username' => $username,
        ]);
    }

    public function getProducts(Request $request, User $user)
    {
        $product = $request->header('product');
        $products = $user->products()->whereNot('products.product_id', $product)->get();
        // return response()->json(['data' => $products]);
        return response()->json(['data' => ProductResource::collection($products)]);
    }
}
