<?php

namespace App\Models;

use App\Enums;
use App\Enums\ActiveEnum;
use App\Enums\VerifiedEnum;
use Illuminate\Support\Str;
use App\Constants\UserConstants;
use Laravel\Passport\HasApiTokens;
use Laravel\Socialite\AbstractUser;
use libphonenumber\PhoneNumberUtil;
use Illuminate\Support\Facades\Hash;
use libphonenumber\PhoneNumberFormat;
use Illuminate\Notifications\Notifiable;
use Illuminate\Notifications\Notification;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;

class User extends Authenticatable
{
    use HasFactory, Notifiable, HasApiTokens;

    /**
     * The primary key for the model.
     *
     * @var string
     */
    protected $primaryKey = 'user_id';

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'username',
        'nickname',
        'phone',
        'email',
        'password',
        'image',
        'cover',
        'timezone',
        'dark_mode',
        'gender',
        'bdate',
        'verification',
        'product_id',
        'fname_ar',
        'lname_ar',
        'fa_name_ar',
        'fname_en',
        'lname_en',
        'fa_name_en',
        'npassword',
        'active_key',
        'semail',
        'company',
        'job',
        'en_job',
        'en_company',
        'secret_a',
        'fax',
        'mobile',
        'post_box',
        'work_phone',
        'website',
        'active',
        'na_number',
        'en_address',
        'ar_address',
        's_code',
        'token_id',
        'en_caddress',
        'ar_caddress',
        // 'en_aboutme',
        // 'ar_aboutme',
        'pincode',
        'pass_code',
        'active_code',
        'time_zone',
        'ann_id',
        'edu_id',
        'public_key',
        'verification_code',
        'verification_code_expires_at',
        'verified',
        'close',
        'city',
        'password',
        'show_contact',
        'username_changes'
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'npassword',
        'remember_token',
    ];


    public static function boot()
    {
        parent::boot();

        self::creating(function ($model) {
            if (isset($model->mobile)) {
                $phoneNumberUtil = PhoneNumberUtil::getInstance();
                $parsedNumber = $phoneNumberUtil->parse($model->mobile);
                $model->mobile = $phoneNumberUtil->format($parsedNumber, PhoneNumberFormat::INTERNATIONAL);
            }
        });


        self::updating(function ($model) {
            if (isset($model->mobile)) {
                $phoneNumberUtil = PhoneNumberUtil::getInstance();
                $parsedNumber = $phoneNumberUtil->parse($model->mobile);
                $model->mobile = $phoneNumberUtil->format($parsedNumber, PhoneNumberFormat::INTERNATIONAL);
            }
        });
    }

    public static function findOrCreateUser(AbstractUser $providerUser, $provider)
    {
        $socialIdentity = SocialIdentity::where('provider_name', $provider)
            ->where('provider_id', $providerUser->getId())
            ->first();

        if ($socialIdentity) {
            return $socialIdentity->user;
        }

        $user = self::where('email', $providerUser->getEmail())->first();

        if (!$user) {
            $user = self::create([
                'username' => $providerUser->getNickname(),
                'email' => $providerUser->getEmail(),
                'password' => Hash::make(Str::random()),
                'name' => $providerUser->getName(),
                'verification' => Enums\VerificationEnum::EMAIL,
            ]);
        }

        $user->identities()->create([
            'provider_id' => $providerUser->getId(),
            'provider_name' => $provider,
        ]);

        return $user;
    }

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'verification_code_expires_at' => 'datetime',
            'password' => 'hashed',
            'gender' => Enums\GenderEnum::class,
            'verification' => Enums\VerificationEnum::class,
            'dark_mode' => Enums\DarkModeEnum::class,
        ];
    }

    public function scopeFilter($query, array $filters)
    {
        if (isset($filters['keyword'])) {
            $keyword = $filters['keyword'];
            $query->Where(function ($q) use ($keyword) {
                $q->WhereAny(
                    ['username', 'email', 'fname_ar', 'lname_ar', 'fa_name_ar', 'fname_en', 'lname_en', 'fa_name_en'],
                    'like',
                    "%{$keyword}%"
                );
            });
        }

        foreach ($filters as $key => $value) {
            if (in_array($key, ['verification_type', 'gender', 'time_zone', 'dark_mode'])) {
                $query->Where($key, $value);
            } else {
                if ($key == 'bdate_start') {
                    $query->whereDate('bdate', '>', $value);
                } else {
                    if ($key == 'bdate_end') {
                        $query->whereDate('bdate', '<', $value);
                    } else {
                        if ($key == 'country') {
                            $query->whereHas('country', function ($query) use ($value) {
                                $query->where('countries.co_id', $value);
                            });
                        }
                    }
                }
            }
        }

        return $query;
    }

    public static function register($data, $language)
    {
        $product_id = $data['product'];
        unset($data['product']);
        unset($data['image']);
        unset($data['cover']);
        $user = new static();

        if ($data['fname'] && $data['lname']) {
            if (app()->getLocale() == 'ar') {
                $data['fname_ar'] = $data['fname'];
                $data['lname_ar'] = $data['lname'];
            } else {
                $data['fname_en'] = $data['fname'];
                $data['lname_en'] = $data['lname'];
            }
        }


        $user->fill($data)->save();

        $user->products()->attach($product_id, [
            'active' => ActiveEnum::YES->value,
            'ldate' => now(),
            'date' => now(),
            'language' => $language ?? app()->getLocale(),
        ]);

        return $user;
    }

    public function sLinks(): HasMany
    {
        return $this->hasMany(related: SLink::class, foreignKey: 'user_id', localKey: 'user_id');
    }

    public function logins()
    {
        return $this->hasMany(Login::class, 'user_id', 'user_id');
    }

    public function image()
    {
        return $this->hasOne(Image::class, 'user_id', 'user_id');
    }

    public function cover()
    {
        return $this->hasOne(Image::class, 'user_id', 'user_id');
    }

    public function covers(): HasMany
    {
        return $this->hasMany(Image::class, 'user_id')
            ->where('type', UserConstants::IMAGE_TYPE_COVER);
    }

    public function photos(): HasMany
    {
        return $this->hasMany(Image::class, 'user_id')
            ->where('type', UserConstants::IMAGE_TYPE_PROFILE);
    }


    public function products(): BelongsToMany
    {
        return $this->belongsToMany(Product::class, 'uproducts', 'user_id', 'product_id')
            ->using(ProductUser::class)
            ->withPivot('active', 'ldate', 'date', 'language', 'token');
    }

    public function identities(): HasMany
    {
        return $this->hasMany(SocialIdentity::class, 'user_id', 'user_id');
    }

    public function cityy()
    {
        return $this->belongsTo(City::class, 'city', 'ci_id');
    }

    public function country()
    {
        return $this->hasOneThrough(Country::class, City::class, 'co_id', 'co_id', 'city', 'co_id');
    }

    public function language()
    {
        return $this->hasOne(Language::class, 'lan_id', 'lang');
    }

    public static function isVerfied($email)
    {
        $user = self::where('email', $email)->first();

        if ($user) {
            return ($user->verified == VerifiedEnum::YES->value);
        }

        return false;
    }

    public function routeNotificationForVonage(Notification $notification): string
    {
        return $this->phone;
    }
}
